//******************************************************************************
//  MSP430x26x Demo - USCI_B0 I2C Slave TX single bytes to MSP430 Master
//
//  Description: This demo connects two MSP430's via the I2C bus. The master
//  reads from the slave. This is the slave code. The TX data begins at 0
//  and is incremented each time it is sent. An incoming start condition
//  is used as a trigger to increment the outgoing data. The master checks the
//  data it receives for validity. If it is incorrect, it stops communicating
//  and the P1.0 LED will stay on. The USCI_B0 TX interrupt is used to know
//  when to TX.
//  ACLK = n/a, MCLK = SMCLK = default DCO = ~1.045MHz
//
//                                 /|\  /|\
//                 MSP430F2619     10k  10k    MSP430F2619
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1         P1.0|-->LED
//             |                 |  |      |                 |
//             |             P3.2|<-+----->|P3.2             |
//             |                 |         |                 |
//             |             P2.0|<--------|P4.7             |
//
//  H. Grewal
//  Texas Instruments Inc.
//  July 2009
//  Built with IAR Embedded Workbench Version: 4.11B
//******************************************************************************
#include "msp430x26x.h"

typedef enum {
             SMBS_MISC               = (unsigned char) 0x04,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/	 
                 

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);

unsigned char TXData;
unsigned char RXData, count = 0;

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P1DIR |= 0x01;                            // LED output
  P2DIR = 0xFE;                             // Set P2.0 to input  
  P2IES = 0;                                // Interrupt edge select for low to high 
  P2IFG = 0;                                // Clear pending P2 interrupts
  P2IE = 0x01;                              // Enable P2.0 interrupt
  P3SEL |= 0x06;                            // Assign I2C pins to USCI_B0
  SMBS_Init (SMBS_MISC,0x48);               // Own address configured as 48h
  
  TXData = 0;                               // Used to hold TX data
  __bis_SR_register(LPM4_bits+GIE);         // Enter LPM4 w/ interrupts
  
}


void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  UCB0CTL1 |= UCSWRST;                      // Enable SW reset
  UCB0CTL0 = UCMODE_3 + UCSYNC;             // I2C Slave, synchronous mode
  if (Add_Param == SMBS_MISC)
  {
      UCB0I2COA = Slave_Add;                // Own Address is 048h
  } 
  else
      UCB0I2COA = (unsigned char) Add_Param ;
  UCB0CTL1 &= ~UCSWRST;                     // Clear SW reset, resume operation  
  IE2 |= UCB0RXIE + UCB0TXIE;               // Enable RX, TX interrupt
}

// USCI_B0 Data ISR
#pragma vector = USCIAB0TX_VECTOR
__interrupt void USCIAB0TX_ISR(void)
{
  if (IFG2 & UCB0RXIFG)                     // If RX Interrupt
  {
    RXData = UCB0RXBUF;                     // Read UCBxRXBUF
  }
  else
  {
    if (count == 0)                         // 1st Byte TX
    {
      IE2 &= ~UCB0TXIE;                     // Read the comment below
      /* Comment the line above if normal Slave operation is desired where
      2 bytes are Transmitted by Slave without the timeout condition.
      If the above line is left uncommented, the 2nd byte will NOT be
      transmitted simulating a delay by the slave to hold the SCL line low and 
      a timeout will occur leading to a port interrupt */
      UCB0TXBUF = TXData++;                 // TX data
      count++;
    }
    else
    {
    __delay_cycles(200);                    // Normal delay between bytes
    UCB0TXBUF = TXData++;                   // TX data
    }
  }
  
}

#pragma vector=PORT2_VECTOR
__interrupt void p2_isr(void)
{
  IE2 &= ~(UCB0TXIE + UCB0RXIE);            // Disable Interrupts
  IFG2 &= ~(UCB0TXIFG + UCB0RXIFG);         // Clear Flags
  P2IFG = 0;                                // Manually Clear port interrupt
  UCB0TXBUF = 0;                            // TX Dummy data to release line
  RXData = UCB0RXBUF;                       // RX Dummy data to release line
  P1OUT ^= 0x01;                            // Toggle LED
  SMBS_Init (SMBS_MISC,0x48);               // Reset Slave
  count = 0;                                // Reset the counter for 1st byte TX
}